;;; Verum-Dezyne --- An IDE for Dezyne
;;;
;;; Copyright © 2020, 2024 Janneke Nieuwenhuizen <janneke@gnu.org>
;;;
;;; This file is part of Verum-Dezyne.
;;;
;;; Verum-Dezyne is property of Verum Software Tools BV <support@verum.com>.
;;; All rights reserved.

;;
;;; Commentary:
;;;
;;; Code:

(define-module (ide commands exec)
  #:use-module (ice-9 getopt-long)
  #:use-module (ice-9 match)
  #:use-module (ide command-line)
  #:use-module (ide util))

(define (parse-opts args)
  (let* ((option-spec
          '((force (single-char #\f))
            (help (single-char #\h))
            (import (single-char #\I) (value #t))
            (model (single-char #\m) (value #t))
            (verbose (single-char #\v))))
	 (options (getopt-long args option-spec #:stop-at-first-non-option #t))
	 (help? (option-ref options 'help #f))
	 (files (option-ref options '() '()))
         (usage? (and (not help?) (null? files))))
    (when (or help? usage?)
      (let ((port (if usage? (current-error-port) (current-output-port))))
        (format port "\
Usage: ide exec [OPTION]... COMMAND-LINE
Run any COMMAND-LINE

  -h, --help             display this help and exit
  -v, --verbose          be verbose; show command to be executed
")
        (exit (or (and usage? EXIT_OTHER_FAILURE) EXIT_SUCCESS))))
    options))

(define (main args)
  (let* ((options (parse-opts args))
         (debug? (ide:command-line:get 'debug))
         (command-line (option-ref options '() '()))
         (verbose? (or (command-line:get 'verbose)
                       (ide:command-line:get 'debug)
                       (ide:command-line:get 'verbose)))
         (local-result EXIT_SUCCESS))

    (match command-line
      ((command args ...)
       (when debug?
         (format (current-error-port) "exec: ~a\n" (string-join command-line)))
       (apply execlp command command-line)))))
