;;; Verum-Dezyne --- An IDE for Dezyne
;;;
;;; Copyright © 2021 Jan (janneke) Nieuwenhuizen <janneke@gnu.org>
;;;
;;; This file is part of Verum-Dezyne.
;;;
;;; Verum-Dezyne is property of Verum Software Tools BV <support@verum.com>.
;;; All rights reserved.

(define-module (ide shell-util)
  #:use-module (srfi srfi-26)
  #:use-module (ice-9 ftw)
  #:use-module (ice-9 match)
  #:use-module (ice-9 regex)
  #:export (directory-exists?
            file-name-extension
            list-directory
            mingw?
            mkdir-p))

(define (mingw?)
  (string-suffix? "mingw32" %host-type))

(define (directory-exists? dir)
  "Return DIR if it exists and is a directory."
  (let ((s (stat dir #f)))
    (and s
         (eq? (stat:type s) 'directory)
         s)))

(define (file-name-extension file-name)
  "Return the extension of FILE-NAME or #f if it does not exist."
  (let ((pos (string-rindex file-name #\.)))
    (and pos (substring file-name pos (string-length file-name)))))

(define* (list-directory dir #:optional (predicate identity))
  "Run SCANDIR on dir, using PREDICATE, and prepend DIR to results."
  (let ((xdir (if (string-suffix? "/" dir) dir
                  (string-append dir "/"))))
    (map (cute string-append dir <>)
         (scandir dir predicate))))

(define (mkdir-p dir)
  "Create directory DIR and all its ancestors."

  (define not-file-name-separator
    (if (mingw?) (char-set-complement (char-set #\/ #\\))
        (char-set-complement (char-set #\/))))

  (let* ((components (string-tokenize dir not-file-name-separator))
         (mingw-prefix? (and (mingw?)
                             (pair? components)
                             (string-suffix? ":" (car components))))
         (root  (if (absolute-file-name? dir)
                    (if mingw-prefix? (car components)
                        "")
                    "."))
         (components (if mingw-prefix? (cdr components) components)))
    (let loop ((components components)
               (root root))
      (match components
        ((head tail ...)
         (let ((dir (string-append root "/" head))) ;use "/" internally
           (catch 'system-error
             (lambda ()
               (mkdir dir)
               (loop tail dir))
             (lambda args
               (if (= EEXIST (system-error-errno args))
                   (loop tail dir)
                   (apply throw args))))))
        (() #t)))))
